
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  Biphase.c: S/PDIF biphase signal generation (simplified from earlier versions as this signal never leaves the PCB)
  Written by Nicholas Vinen, 2012-2013
*/

#include "Biphase.h"
#include <string.h>

#define PREAMBLE_X 0xE200
#define PREAMBLE_Y 0xE400
#define PREAMBLE_Z 0xE800

static unsigned short biphase_encode_8[256];
static unsigned short biphase_encode_4[16], biphase_encode_4_Y[16];
static unsigned char parity_table[256];

void Biphase_Generate_Tables() {
  unsigned short i, j, s, p;

  for( i = 0; i < 256; ++i ) {
    p = 0;
    s = 0;
    for( j = 0; j < 8; ++j ) {
      s <<= 2;
      if( i&(1<<j) ) {
        p ^= 0x80;
        s |= s & 4 ? 1 : 2;
      } else {
        s |= s & 4 ? 0 : 3;
      }
    }
    parity_table[i] = p;
    biphase_encode_8[i] = s;
    if( i < 16 ) {
      biphase_encode_4[i] = s>>8;
      biphase_encode_4_Y[i] = (s>>8) | PREAMBLE_Y;
	}
  }
}

static unsigned short polarity, subframe;
static inline unsigned short getval16(unsigned short val) {
    unsigned short ret = val ^ polarity;
    polarity = ((((signed short)ret)<<15)>>15);
    return ret;
}

void Biphase_Encode_16bit(unsigned short* dst, const signed short* src, unsigned char num_samples) {
  const unsigned short* ssrc = (const unsigned short*)src;
  unsigned short preambleL = (subframe == 0 ? PREAMBLE_Z : PREAMBLE_X) | 0xCC;
  do {
    dst[0] = getval16( preambleL );
    dst[1] = getval16( biphase_encode_8[(ssrc[0]<<4)&0xFF] );
    dst[2] = getval16( biphase_encode_8[(ssrc[0]>>4)&0xFF] );
    dst[3] = getval16( biphase_encode_8[(ssrc[0]>>12) | (parity_table[(ssrc[0]&0xFF)^(ssrc[0]>>8)])] );
    dst += 4;
    ssrc += 1;

    dst[0] = getval16( PREAMBLE_Y | 0xCC );
    dst[1] = getval16( biphase_encode_8[(ssrc[0]<<4)&0xFF] );
    dst[2] = getval16( biphase_encode_8[(ssrc[0]>>4)&0xFF] );
    dst[3] = getval16( biphase_encode_8[(ssrc[0]>>12) | (parity_table[(ssrc[0]&0xFF)^(ssrc[0]>>8)])] );
    dst += 4;
    ssrc += 1;

	++subframe;
	preambleL = PREAMBLE_X | 0xCC;
  } while( --num_samples );

  if( subframe == 192 )
    subframe = 0;
}

void Biphase_Encode_24bit(unsigned short* dst, const signed short* src, unsigned char num_samples) {
  const unsigned short* ssrc = (const unsigned short*)src;
  unsigned short preambleL = (subframe == 0) ? PREAMBLE_Z : PREAMBLE_X;
  unsigned short t = polarity;

  do {
    t ^= ( preambleL | biphase_encode_4[ssrc[0]&0xF] );
	dst[0] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[((ssrc[0]>>4)&0xFF)] );
	dst[1] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[((ssrc[0]>>12)|((ssrc[1]&0xF)<<4))] );
	dst[2] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[((ssrc[1]>>4)&0xF) | (parity_table[((ssrc[0]^ssrc[1])&0xFF)^(ssrc[0]>>8)])] );
	dst[3] = t;
    dst += 4;

	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_4_Y[(ssrc[1]>>8)&0xF] );
	dst[0] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[(ssrc[1]>>12)|((ssrc[2]&0xF)<<4)] );
	dst[1] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[((ssrc[2]>>4))&0xFF] );
	dst[2] = t;
	t = ((((signed short)t)<<15)>>15);
    t ^= ( biphase_encode_8[(ssrc[2]>>12) | (parity_table[((ssrc[1]^ssrc[2])>>8)^(ssrc[2]&0xFF)])] );
	dst[3] = t;
	t = ((((signed short)t)<<15)>>15);
    dst += 4;
    ssrc += 3;
/*  Above version seems to be faster.
    dst[0] = getval16( preambleL | biphase_encode_4[ssrc[0]&0xF] );
    dst[1] = getval16( biphase_encode_8[((ssrc[0]>>4)&0xFF)] );
    dst[2] = getval16( biphase_encode_8[((ssrc[0]>>12)|((ssrc[1]&0xF)<<4))] );
    dst[3] = getval16( biphase_encode_8[((ssrc[1]>>4)&0xF) | (parity_table[((ssrc[0]^ssrc[1])&0xFF)^(ssrc[0]>>8)])] );
    dst += 4;

    dst[0] = getval16( biphase_encode_4_Y[(ssrc[1]>>8)&0xF] );
    dst[1] = getval16( biphase_encode_8[(ssrc[1]>>12)|((ssrc[2]&0xF)<<4)] );
    dst[2] = getval16( biphase_encode_8[((ssrc[2]>>4))&0xFF] );
    dst[3] = getval16( biphase_encode_8[(ssrc[2]>>12) | (cs^parity_table[((ssrc[1]^ssrc[2])>>8)^(ssrc[2]&0xFF)])] );
    dst += 4;
    ssrc += 3;
*/
	preambleL = PREAMBLE_X;
  } while( --num_samples );

  polarity = t;

  if( subframe == 192 )
    subframe = 0;
}
